<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

use App\Events\VoucherPayment;

use App\Http\Requests;

use App\ExamsVouchers;

use Auth;

use Gate;

use Validator;

class ExamsVoucherController extends Controller
{

    public function __construct(){

    	$this->middleware('auth:super_admin', ['except' => ['cardPurchase', 'paystackRedirectToGateway', 'handlePaystackGatewayCallback']]);
    }

    protected function permissionDeny($ability){
        Auth::shouldUse('super_admin');
        return Gate::denies($ability);
    }


     public function viewScratchCard()
     {
     	if($this->permissionDeny('view-scratch-card')){
            $title = 'Permission Deny';
            return view('errors.401', compact('title'));
        }

         $title = "Vouchers";
         
         
         if (\Request::ajax())
         {
             return \Response::json(ExamsVouchers::all());
         }
         
         $vouchers = ExamsVouchers::paginate(100);

         $iterations = array(1 ,2 ,5, 10, 20, 50, 100, 200);

         $exam_types = array('Unity Exam', 'Junior WAEC', 'Pre WAEC', 'Entrance');
         
         
         return view('super-admin.exam-voucher', compact('title','vouchers', 'iterations', 'exam_types'));
     }
     
     /**
      * Create Vouchers
      * @params Request
      * @return Response
      */ 
     public function generateScratchCard(Request $request)
     {
     	if($this->permissionDeny('create-scratch-card')){
            $title = 'Permission Deny';
            return view('errors.401', compact('title'));
        }

        $voucher = new ExamsVouchers;

         $validator = Validator::make($request->all(), $voucher::$rules);
         
         if ($validator->passes())
         {
             
            $iteration = $request->input('iteration');

            $insert = array();

            for ($i=1; $i <= $iteration; $i++)
            {                 
                list($pin1, $pin2, $pin3, $pin4) = $this->getRandomVoucher();

                $voucher->create([
                    'pin1' => $pin1,
                    'pin2' => $pin2,
                    'pin3' => $pin3,
                    'pin4' => $pin4,
                    'pin' => $pin1 . $pin2 . $pin3 . $pin4,
                   'serial' => $this->getSerialNumber(),
                   'multiple' => $request->input('multiple'),
                   'exam_type' => $request->input('exam_type'),
                ]);
            }
             
            flash('Scratch Cards generated!')->success();
            return redirect()->back();
         }
         
         flash()->warning('Something went wrong');
         return redirect()->back()
                           ->withErrors($validator)
                           ->withInput();
     }

     public function deleteScratchCard(Request $request){

     	if($this->permissionDeny('delete-scratch-card')){
            $title = 'Permission Deny';
            return view('errors.401', compact('title'));
        }

     	$validate = Validator::make($request->all(), [
     						'scratch_cards' => 'required|array'
     		]);

     	if($validate->passes()){

     		$vouchers_to_delete = '('.implode(', ', $request->scratch_cards).')';

     			$vouchers = ExamsVouchers::whereRaw('`id` IN '.$vouchers_to_delete)
     						->delete();

     			flash('You have successfully deleted the selected scratch cards')->success();

     			return redirect()->back();
     	}

     	flash('Something is wrong')->error();

     	return redirect()->back()
     					->withErrors($validate)
     					->withInput();

     }

     public function cardPurchase(){
        $title = 'Purchase Examination Card Online';

        return view('exams.purchase-card', compact('title'));
     }

      public function paystackRedirectToGateway(Request $request){

        $validate = Validator::make($request->all(), [
                'email' => 'required|email',
                'fullname' => 'required|min:5',
                'exam_type' => ['required','regex:/(unity_exam|entrance)/'],
                'phone' => 'sometimes|regex:(234?)|digits:13',
                'slot' => 'required|integer|min:1'
            ]);

        if($validate->passes()){

                $paystack = new \Yabacon\Paystack('test' == getenv('PAYSTACK_MODE') ? getenv('PAYSTACK_TEST_SECRET_KEY') : getenv('PAYSTACK_LIVE_SECRET_KEY'));

                switch ($request->input('exam_type')) {
                    case 'unity_exam':
                        $amount = 3500 * $request->input('slot');
                    break;
                    case 'entrance':
                        $amount = 2000 * $request->input('slot');
                    break;
                }
                //Building metadata :: More information about the transaction
                $builder = new \Yabacon\Paystack\MetadataBuilder();
                $builder->withCustomField('Fullname', $request->input('fullname'));
                $builder->withCustomField('Exam', $request->input('exam_type'));
                $builder->withCustomField('Phone', $request->input('phone'));
                $builder->withCustomField('Slot', $request->input('slot'));
                $metadata = $builder->build();

                try
                {
                  $tranx = $paystack->transaction->initialize([
                    'amount'=> $amount * 100,       // in kobo
                    'email'=> $request->input('email'),         // unique to customers
                    'reference' => $this->getSerialNumber(), // unique to transactions
                    'first_name' => $request->input('fullname'),
                    'metadata' => $metadata,
                  ]);
                } catch(\Yabacon\Paystack\Exception\ApiException $e){
                  print_r($e->getResponseObject());
                  die($e->getMessage());
                }

                // dd($tranx);

                return redirect($tranx->data->authorization_url);
         }

        flash('Something is wrong')->error();

        return redirect()->back()
                        ->withErrors($validate)
                        ->withInput();

    }

    public function handlePaystackGatewayCallback(Request $request){
        $reference = isset($_GET['reference']) ? $_GET['reference'] : '';
        if(!$reference){
          die('No reference supplied');
        }

        $paystack = new \Yabacon\Paystack('test' == getenv('PAYSTACK_MODE') ? getenv('PAYSTACK_TEST_SECRET_KEY') : getenv('PAYSTACK_LIVE_SECRET_KEY'));
        try
        {
          // verify using the library
          $tranx = $paystack->transaction->verify([
            'reference'=>$reference, // unique to transactions
          ]);
        } catch(\Yabacon\Paystack\Exception\ApiException $e){
          print_r($e->getResponseObject());
          die($e->getMessage());
        }

        if ('success' === $tranx->data->status) {

            $voucher = new ExamsVouchers;

            if($voucher->where('serial', '=', $tranx->data->reference)->count() == 0){

                $user = new \stdClass;
                $more_details = $tranx->data->metadata;

                foreach($more_details->custom_fields as $custom_field){
                    if(strtolower($custom_field->variable_name) == 'fullname' ){
                        $user->fullname = $custom_field->value;
                    }elseif(strtolower($custom_field->variable_name) == 'exam' ){
                        $user->exam = $custom_field->value;
                    }elseif(strtolower($custom_field->variable_name) == 'slot' ){
                        $user->slot = $custom_field->value;
                    }elseif(strtolower($custom_field->variable_name) == 'phone' ){
                        $user->phone = $custom_field->value;
                    }
                    
                }

                list($pin1, $pin2, $pin3, $pin4) = $this->getRandomVoucher();

                $voucher->create([
                    'pin1' => $pin1,
                    'pin2' => $pin2,
                    'pin3' => $pin3,
                    'pin4' => $pin4,
                    'pin' => $pin1 . $pin2 . $pin3 . $pin4,
                   'serial' => $tranx->data->reference,
                   'multiple' => $user->slot,
                   'exam_type' => $user->exam
                ]);
                //Creating request data
                $user->serial_no = $tranx->data->reference;
                $user->pin = $pin1 . $pin2 . $pin3 . $pin4;
                $user->email = $tranx->data->customer->email;
                $user->amount = (($tranx->data->amount) / 100);
                //Converting the metadata to request variable

                $message = urlencode("Information of Voucher. PIN = ".$user->pin." and Serial = ". $user->serial_no.".Check ".$user->email." for more details");

                prepareSMSGateWay(urlencode('ODSGMOE'), urlencode($user->phone), $message);

                event(new VoucherPayment($user));

            }
            flash('You have suucessfully purchase the voucher, check your mail or phone the pin and serial no')->success();

            return redirect()->route('exam.voucher.purchase');

        }


        flash('Transaction failed. Please try again')->error();

        return redirect()->route('exam.voucher.purchase');

    }
     
     /**
      * Generates Random Vouchers
      * @params int $iteration
      * @params int $amount
      * @return Array Vouchers
      */ 
     private function getRandomVoucher()
     {
         $pin1 = rand(1111,9999);
         $pin2 = rand(1111,9999);
         $pin3 = rand(1111,9999);
         $pin4 = rand(1111,9999);
         
         
         $usedPins = ExamsVouchers::where('pin1', $pin1)
                                ->where('pin2', $pin2)
                                ->where('pin3', $pin3)
                                ->where('pin4', $pin4)->get();
                                
         if ($usedPins->count() > 0)
         {
             return $this->getRandomVoucher();
             
         }
         else
         { 
            return array($pin1, $pin2, $pin3, $pin4);
         }
           
     }

     private function getSerialNumber(){
     	$serial = 'SN'.rand(11111111, 99999999);

     	$usedPins = ExamsVouchers::where('serial', $serial)->get();
                                
         if ($usedPins->count() > 0)
         {
             return $this->getSerialNumber();
             
         }
         else
         { 
            return $serial;
         }
     }
}
