<?php

namespace App;

use Illuminate\Foundation\Auth\User as Authenticatable;

class Teacher extends Authenticatable
{
    protected $table = 'teachers';

    protected $fillable = [
       'surname',
        'firstname',
        'middlename',
        'school_id',
        'staff_no',
        'staff_no_digit',
        'qualification',
        'gender',
        'address',
        'email',
        'phone',
        'session',
        'state_id',
        'lga_id',
        'subjects',
        'password',
        'next_of_kins',
        'next_of_kins_address',
        'next_of_kins_phone',
        'next_of_kins_email',
        'health_status',
        'extra_curricular_activites',
        'health_status_desc' ,
        'marital_status',
        'title'
    ];

    protected $guard = 'teacher';

     public static $rulesLogin = [
        'staff_id' => 'required',
        'password' => 'required',
    ];

    protected $hidden = ['password', 'remember_token'];

    public static $rules = [
        'title' => 'required|min:2',
		'surname' => 'required|min:3',
		'firstname' => 'required|min:3',
		'qualification' => 'required',
		'gender' => ['required','regex:/^(Male|Female)/'],
        'marital_status' => ['required','regex:/^(Single|Married|Divorce)/'],
		'address' => 'required|min:10',
		'email' => 'required|email|unique:teachers,email',
		'phone' => 'required|regex:(234?)',
        'session' => 'required|integer|digits:4',
		'state_id' => 'required|integer',
		'lga_id' => 'required|integer',
		'subjects' => 'required|array|min:1',
		'password' => 'required|alpha_num|min:6|confirmed',
		'password_confirmation' => 'required|alpha_num|min:6',
		'next_of_kins' => 'required|min:10',
		'next_of_kins_address' => 'required|min:10',
		'next_of_kins_phone' => 'required|regex:(234?)',
        'next_of_kins_email' => 'sometimes|email',
		'health_status' => ['required', 'regex:/(Normal|Disable)/'],
		'health_status_desc' => 'required|min:3'
	];

     public static $teacherProfileRule = [
        'title' => 'required|min:2',
        'surname' => 'required|min:3',
        'firstname' => 'required|min:3',
        'middlename' => 'sometimes|min:3',
        'gender' => ['required','regex:/(Male|Female)/'],
        'marital_status' => ['required','regex:/^(Signle|Married|Divorce)/'],
        'address' => 'required|min:10',
        'state_id' => 'required|integer',
        'lga_id' => 'required|integer',
        'phone' => 'required|regex:(234?)|digits:13',
        'qualification' => 'required',
        'next_of_kins' => 'required|min:10',
        'next_of_kins_address' => 'required|min:10',
        'next_of_kins_phone' => 'required|regex:(234?)',
        'next_of_kins_email' => 'sometimes|email',
        'health_status' => ['required', 'regex:/(Normal|Disable)/'],
        'health_status_desc' => 'required|min:3'
    ];

    public static $teacherPassportRule = [
        'passport' => 'required|image|mimes:jpeg,jpg,png|img_min_size:250'
    ];

    public static $teacherPasswordRule = [
        'password' => 'required|alpha_num|min:6|confirmed',
        'password_confirmation' => 'required|alpha_num|min:6',
    ];

    public static $teacherSubjectAreaRule = [
        'subjects' => 'required|array|min:1',
    ];

    public static $ruleBatch = [
        'session' => 'required|digits:4',
        'pseudo_batch_file_name' => ['required','allowexts:xls,xlsx,csv', 'excelFormatAllowed:18']
    ];

    public function subjects(){

        return $this->belongsToMany('App\Subject', 'subject_teacher', 'teacher_id', 'subject_id')->withPivot('school_id');

    }

    public function permission_classes(){

        return $this->belongsToMany('App\Permission', 'class_teacher_permission', 'teacher_id', 'permission_id')->withPivot('classarm_id', 'subject_id');

    }

    public function houses(){

        return $this->belongsToMany('App\StudentHouse', 'house_house_master', 'teacher_id', 'house_id')->withPivot('school_id','session');

    }

    public function state(){

    	return $this->belongsTo('App\NgStates', 'state_id');

    }

    public function statelga(){

    	return $this->belongsTo('App\NgStatesLGA', 'lga_id');

    }

    public function classarms(){

        return $this->belongsToMany('App\ClassArms', 'classarm_teacher', 'teacher_id', 'classarm_id')->withPivot('session');

    }

    public function counsellors(){

        return $this->belongsToMany('App\ClassArms', 'classarm_counsellor', 'teacher_id', 'classarm_id')->withPivot('session');

    }

    public function hasPermission($permission){
        $permissions = array();
        if($this->classarms()->get()->count() > 0){
            foreach ($this->permission_classes()->wherePivot('classarm_id','=',$this->classarms()->get()->toArray()[0]['id'])->get() as $permits) {
                $permissions[] = $permits->permission_slug;
            }

            return in_array($permission, $permissions);
            
        }

        return false;

    }

    public function schools(){
        return $this->belongsTo('\App\School', 'school_id');
    }

    public function school(){
        return $this->belongsTo('App\School', 'school_id');
    }
    
    public function attendanceAnalytic(){
        return $this->hasMany('App\AttendanceAnalytic', 'teacher_id');
    }
    
    public function attendanceAnalyticLog(){
        return $this->hasMany('App\AttendanceAnalyticLog', 'teacher_id');
    }
}
