<?php

namespace App;

use Illuminate\Foundation\Auth\User as Authenticatable;

class Student extends Authenticatable
{
    protected $table = 'students';


    protected $fillable = [
        'surname',
        'firstname',
        'middlename',
        'regnum',
        'regnum_digit',
        'school_id',
        'dob',
        'gender',
        'country',
        'address',
        'state_id',
        'lga_id',
        'religion',
        'password',
        'password_confirmation',
        'parent_fullname',
        'parent_address',
        'parent_email',
        'parent_phone',
        'session',
        'house_id',
        'blood_group'
    ];


    protected $guard = 'student';

    public static $rulesLogin = [
        'regnum' => 'required',
        'password' => 'required',
        //'school' => 'required'
    ];

    protected $hidden = ['password', 'remember_token'];

    public static $rules = [
		'surname' => 'required|min:3',
		'firstname' => 'required|min:3',
		'dob' => 'required|date',
		'gender' => ['required','regex:/(Male|Female)/'],
		'country' => 'required',
		'address' => 'required|min:10',
		'state_id' => 'required|integer|not_in:0',
		'lga_id' => 'required|integer|not_in:0',
        'house_id' => 'required|integer|not_in:0',
		'religion' => ['required','regex:/(Christian|Muslim|Traditional)/'],
		'password' => 'required|alpha_num|min:6|confirmed',
		'password_confirmation' => 'required|alpha_num|min:6',
		'parent_fullname' => 'required',
		'parent_address' => 'required',
		'parent_email' => 'required|uniqueFirstAndLastName:surname,firstname,session',
		'parent_phone' => 'required|regex:(234?)|digits:13',
		'session' => 'required|digits:4',
		'term' => ['required','regex:/(First|Second|Third)/'],
		'class_id' => 'required|integer',
		'class_arm_id' => 'required|integer',
        'phone' => 'sometimes|regex:(234?)|digits:13',
        'blood_group' => ['required','regex:/(A+|A-|B+|B-|AB+|AB-|O+|O-)/'],
	];

    public static $studentProfileRule = [
        'surname' => 'required|min:3',
        'firstname' => 'required|min:3',
        'dob' => 'required|date',
        'gender' => ['required','regex:/(Male|Female)/'],
        'country' => 'required',
        'address' => 'required|min:10',
        'state_id' => 'required|integer',
        'lga_id' => 'required|integer',
        'religion' => ['required','regex:/(Christian|Muslim|Traditional)/'],
         'phone' => 'sometimes|regex:(234?)|digits:13'
    ];

    public static $studentPassportRule = [
        'passport' => 'required|image|mimes:jpeg,jpg,png|img_min_size:250'
    ];

    public static $studentPasswordRule = [
        'password' => 'required|alpha_num|min:6|confirmed',
        'password_confirmation' => 'required|alpha_num|min:6',
    ];

    public static $studentResultSelectRule = [
        'session' => 'required|digits:4',
        'term' => ['required','regex:/(First|Second|Third)/'],
        'class_arm_id' => 'required|integer',
        'class_id' => 'required|integer',
    ];

    public static $ruleBatch = [
		'session' => 'required|digits:4',
		'term' => ['required','regex:/(First|Second|Third)/'],
		'class_id' => 'required|integer',
		'class_arm_id' => 'required|integer',
		'pseudo_batch_file_name' => ['required','allowexts:xls,xlsx,csv', 'excelFormatAllowed:13', 'school_check']
	];

    

    public function state(){

    	return $this->belongsTo('App\NgStates','state_id');

    }

    public function statelga(){

    	return $this->belongsTo('App\NgStatesLGA', 'lga_id');

    }

    public function classes(){

    	return $this->belongsTo('App\Classes', 'class_id');

    }

    public function class_arm(){

    	return $this->belongsTo('App\ClassArms', 'class_arm_id');

    }

     public function classarms(){

        return $this->belongsToMany('App\ClassArms', 'classarm_student', 'student_id', 'classarm_id')->withPivot('session', 'term', 'class_id');

    }

     public function idCardRequest(){

        return $this->hasMany('App\StudentIDCardRequest', 'school_id');

    }

    public function studentResults(){

        return $this->hasMany('App\StudentResult', 'student_id');

    }

    public function studentPracticalSkills(){

        return $this->hasMany('App\PracticalSkill', 'student_id');

    }

    public function studentCharacterAttitudes(){

        return $this->hasMany('App\CharacterAttitude', 'student_id');

    }

    public function StudentResultVouchers(){
        return $this->hasMany('App\ResultVoucher', 'student_id');
    }

    public function schools(){
        return $this->belongsTo('\App\School', 'school_id');
    }

    public function school(){
        return $this->belongsTo('App\School', 'school_id');
    }

    public function studentIDCard(){
        return $this->hasOne('App\StudentIDCard', 'student_id');
    }

    public function beforeTransfer(){
        return $this->hasMany('App\Transfer', 'student_former_id');
    }

    public function afterTransfer(){
        return $this->hasMany('App\Transfer', 'student_new_id');
    }

    public function attendanceAnalyticLog(){
        return $this->hasMany('App\AttendanceAnalyticLog', 'student_id');
    }
    
    public function subjectAttendanceAnalyticLog(){
        return $this->hasMany('App\SubjectAttendanceAnalyticLog', 'student_id');
    }
    
    public function classAttendanceAnalyticLog(){
        return $this->hasMany('App\ClassAttendanceAnalyticLog', 'student_id');
    }
}